//  : aim.
//  Graphics     .

unit u_graphics;
{$mode objfpc}

interface

uses
	u_log, u_options, u_window, u_textures, u_sprites,
	Direct3D9, D3DX9, windows, Classes;

type
	//  .
	TCamera = class
	private
		View: TD3DXMATRIX;
		Pos: TD3DXVECTOR3;
		Up: TD3DXVECTOR3;
		Target: TD3DXVECTOR3;
	public
		constructor Create;
		destructor Destroy; override;
		procedure SetPosition();
		procedure SetZoom();
	end;

	// ,  .
	TGraphics = class(TThread)
	private
		FCamera: TCamera;
		UpdateTime: Double; //     .
		//       Direct3D.
		procedure OnLostDevice;
		procedure OnResetDevice;
		//   .
		procedure RenderScene;
	public
		//   -   .     
		//  .
		constructor Create(const AppName: string);
		destructor Destroy; override;
		//      .
		procedure Execute; override;
		//       .
		property Camera: TCamera read FCamera;
	end;

var
	Graphics: TGraphics = nil;

implementation

//   TD3DWindow,   .
type
	TAppWindow = class(TD3DWindow)
	public
		procedure OnLostDevice; override;
		procedure OnResetDevice; override;
	end;

procedure TAppWindow.OnLostDevice;
begin
	inherited;
	if Graphics <> nil then Graphics.OnLostDevice;
end;

procedure TAppWindow.OnResetDevice;
begin
	inherited;
	if Graphics <> nil then Graphics.OnResetDevice;
end;

// TCamera -----

// private

// public

constructor TCamera.Create;
begin
end;

destructor TCamera.Destroy;
begin
end;

procedure TCamera.SetPosition();
begin
	Pos := D3DXVECTOR3(0.0, 0.0, -1000.0);
	Up := D3DXVECTOR3(0.0, 1.0, 0.0);
	Target := D3DXVECTOR3(0.0, 0.0, 0.0);
	D3DXMatrixLookAtLH(View, Pos, Target, Up);
	Window.Device.SetTransform(D3DTS_VIEW, View);
end;

procedure TCamera.SetZoom();
var
	P: TD3DMATRIX;
	R: TRECT;
begin
	GetClientRect(Window.Handle, R);
	D3DXMatrixPerspectiveFovLH(P, D3DX_PI * 0.25, R.Right / R.Bottom, 1.0, 5000.0);
	Window.Device.SetTransform(D3DTS_PROJECTION, P);
end;

// TGraphics -----

// private

procedure TGraphics.OnLostDevice;
begin

end;

procedure TGraphics.OnResetDevice;
begin
	Camera.SetPosition;
	Camera.SetZoom;
	with Window.Device do begin
		SetSamplerState(0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR);
		SetSamplerState(0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR);
		SetSamplerState(0, D3DSAMP_MIPFILTER, D3DTEXF_LINEAR);
		SetRenderState(D3DRS_LIGHTING, 0);
		SetRenderState(D3DRS_ALPHAREF, 10);
		SetRenderState(D3DRS_ALPHAFUNC, D3DCMP_GREATER);
		SetTextureStageState(0, D3DTSS_ALPHAARG1, D3DTA_TEXTURE);
		SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_SELECTARG1);
		SetRenderState(D3DRS_SRCBLEND, D3DBLEND_SRCALPHA);
		SetRenderState(D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA);
		SetTextureStageState(0, D3DTSS_TEXTURETRANSFORMFLAGS, D3DTTFF_COUNT2);
	end;
end;

procedure TGraphics.RenderScene;
begin
	with Window do begin	
		if not DeviceLost then begin
			//  .
			Device.Clear(0, nil, D3DCLEAR_TARGET or D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0, 0, 255), 1.0, 0);
			// .
			Device.BeginScene;
			Sprites.Draw;
			Device.EndScene;
			//  .
			Device.Present(nil, nil, 0, nil);
		end;
	end;
end;

// public

constructor TGraphics.Create(const AppName: string);
begin
	inherited Create(true);
	UpdateTime := 10; // 100    .
	
	Window := TAppWindow.Create(AppName, AppName);
	Textures := TTextures.Create;
	Sprites := TSprites.Create;
	FCamera := TCamera.Create;
	
	OnResetDevice;
end;

destructor TGraphics.Destroy;
begin
	Window.Destroy;
	Textures.Destroy;
	Sprites.Destroy;
	FCamera.Destroy;
	inherited;
end;

procedure TGraphics.Execute;
var
	CountsPerSec: Int64;
	MSecsPerCount: Double;
	CurrTimeStamp: Int64;
	PrevTimeStamp: Int64;
	Delta: Double;
	SleepTime: Int64;
begin
	Log.Write('Graphics', 'Processing start.');
	QueryPerformanceFrequency(CountsPerSec);
	MSecsPerCount := 1000 / CountsPerSec;
	QueryPerformanceCounter(PrevTimeStamp);
	while Window.Active and not Terminated do begin
		QueryPerformanceCounter(CurrTimeStamp);
		Delta := (CurrTimeStamp - PrevTimeStamp) * MSecsPerCount;
		Sprites.Update(Delta);
		Window.Update;
		RenderScene;
		SleepTime := Round(UpdateTime - Delta);
		PrevTimeStamp := CurrTimeStamp;
		if SleepTime > 0 then Sleep(SleepTime);
	end;
	Log.Write('Graphics', 'Processing stop.');
end;

end.
