program expr;

{$APPTYPE CONSOLE}

{$S+}
type
  tresult = double;

  toperation = (opNone, opAdd, opSubtract, opMult, opDiv);

  texpr = ^tree;
  tree = record
    oper: toperation;
    value: string;
    left, right: texpr;
  end;

  tformula = class
    f: texpr;

    constructor create(s: string);
    function eval: double;

    procedure print;

  private
    procedure printtree(level:integer; node: texpr);

    function check_first(node: texpr): boolean;
    function check_second(node: texpr): boolean;

    procedure change(node: texpr);
  end;


const
  opChar: array[toperation] of char = (
    #0, '+', '-', '*', '/'
  );

function sign_to_operation(ch: char): toperation;
var
  i: toperation;
begin
  for i := low(toperation) to high(toperation) do
    if opChar[i] = ch then begin
      sign_to_operation := i; exit
    end;
  sign_to_operation := opNone;
end;

constructor tformula.create(s: string);
  const
    sLetters     = ['a' .. 'z'];
    sDigits      = ['0' .. '9'];
    sPointDigits = sDigits + ['.'];
    sOperator    = ['+', '-', '*', '/', '^', '(', ')'];
  var
    currP: integer;
  type
    ttoken = (tokNumber, tokOper, tokVar, tokNone);

  function GetToken(var token: string): ttoken;
  begin
    token := ''; GetToken := tokNone;
    if currP > length(s) then exit;

    while s[currP] = ' ' do inc(currP);

    token := '';

    if s[currP] in sDigits then begin
      GetToken := tokNumber;
      while s[currP] in sPointDigits do begin
        token := token + s[currP]; inc(currP);
      end;
    end
    else
      if s[currP] in sOperator then begin
        GetToken := tokOper;
        token := token + s[currP]; inc(currP);
      end
      else
        if s[currP] in sLetters then begin
          GetToken := tokVar;
          token := token + s[currP]; inc(currP);
        end
        else
          GetToken := tokNone;

  end;

var
  token_str: string;
  curr_token: ttoken;

  function Mult: texpr; forward;
  function Factor: texpr; forward;

  function Add: texpr;
  var p, q: texpr;
  begin
    p := Mult;
    while (curr_token = tokOper) and (token_str[1] in ['-', '+']) do begin
      new(q);
      q^.oper := sign_to_operation(token_str[1]);
      curr_token := GetToken(token_str);
      q^.left := p;
      q^.right := Mult;
      p := q;
    end;
    Add := p;
  end;

  function Mult: texpr;
  var p, q: texpr;
  begin
    p := Factor;
    while (curr_token = tokOper) and (token_str[1] in ['*', '/']) do begin
      new(q);
      q^.oper := sign_to_operation(token_str[1]);
      curr_token := GetToken(token_str);
      q^.left := p;
      q^.right := Factor;
      p := q;
    end;
    Mult := p;
  end;

  function Factor: texpr;
  var q: texpr;
  begin
    case curr_token of
      tokNumber, tokVar:
      begin
        new(q);
        q^.oper := opNone;
        q^.value := token_str;
        curr_token := GetToken(token_str);
        q^.left := nil; q^.right := nil;
        Factor := q;
      end;

      tokOper:
      case token_str[1] of
        '(':
        begin
          curr_token := GetToken(token_str);
          Factor := Add;
          curr_token := GetToken(token_str);
        end;
        '-':
        begin
          new(q);
          q^.oper := sign_to_operation(token_str[1]);
          curr_token := GetToken(token_str);
          q^.right := Factor;
          q^.left := nil;

          Factor := q;
        end;
      end;
    end;
  end;

begin
  currP := 1;
  token_str := s;
  curr_token := GetToken(token_str);
  f := Add;

  writeln('parsing finished');
end;

function tformula.eval: Double;

  function Calc(p: texpr): tresult;
  var
    f: tresult;
    err: integer;
  begin
    if p <> nil then
      case p^.oper of
        opAdd      : Calc := Calc(p^.left) + Calc(p^.right);
        opSubtract : Calc := Calc(p^.left) - Calc(p^.right);
        opMult     : Calc := Calc(p^.left) * Calc(p^.right);
        opDiv      : Calc := Calc(p^.left) / Calc(p^.right);

        opNone     :
        begin
          val(p^.value, f, err);
          Calc := f;
        end;
      end
    else Calc := 0;
  end;

begin
  eval := Calc(f);
end;

procedure tformula.printtree(level: integer; node: texpr);
begin
  if node = nil then exit;

  if node^.oper <> opNone then writeln('':2*level, opchar[node^.oper])
  else Writeln('':2*level, node^.value);

  printtree(level+1, node^.left);
  printtree(level+1, node^.right);
end;

procedure tformula.print;
begin
  printtree(0, f);
end;

function tformula.check_first(node: texpr): boolean;
begin
  check_first :=
    (node <> nil) and (node^.oper = opMult) and
    (node^.left <> nil) and (node^.left^.oper = opNone) and
    (node^.right <> nil) and (node^.right^.oper in [opAdd, opSubtract]);
end;
function tformula.check_second(node: texpr): boolean;
begin
  check_second :=
    (node <> nil) and (node^.oper = opMult) and
    (node^.right <> nil) and (node^.right^.oper = opNone) and
    (node^.left <> nil) and (node^.left^.oper in [opAdd, opSubtract]);
end;

procedure tformula.change(node: texpr);

  procedure CopyTree(T: texpr; var T1: texpr);
  begin
    if T = nil then T1 := nil
    else begin
      new(T1);
      T1^.oper := t^.oper; t1^.value := t^.value;
      CopyTree(T^.Left, T1^.Left);
      CopyTree(T^.Right, T1^.Right);
    end
  end;

var q, copied: texpr;
begin
  if node = nil then exit;

  if check_first(node) then begin
    CopyTree(node^.left, copied);

    new(q);
    q^.oper := opMult;
    q^.value := '';
    q^.left := copied;
    q^.right := node^.right^.left;

    node^.left := q;

    node^.oper := node^.right^.oper;
    node^.right.oper := opMult;
    node^.right^.left := copied;
  end
  else
    if check_second(node) then begin
      CopyTree(node^.right, copied);

      new(q);
      q^.oper := opMult;
      q^.value := '';
      q^.right := copied;
      q^.left := node^.left^.right;

      node^.right := q;

      node^.oper := node^.left^.oper;
      node^.left.oper := opMult;
      node^.left^.right := copied;
    end;

  change(node^.left);
  change(node^.right);

end;

var
  s: string;
  ex: tresult;

  fr: tformula;

const
  params: array[1 .. 10] of tresult = (
    0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0
  );

begin
  s := '2+72 -  (2 + 7*(3+8))*1 / 5';
  fr := tformula.create(s);

  fr.print;
  writeln('eval before = ', fr.eval():10:2);

  fr.change(fr.f);
  fr.print;
  writeln('eval after = ', fr.eval():10:2);
  readln;
end.

